/** @file   Barrel.cpp
 * @brief   Implementation of Barrel class.
 * @version $Revision: 1.5 $
 * @date    $Date: 2006/08/13 10:50:22 $
 * @author  Tomi Lamminsaari
 */

#include "Barrel.h"
#include "www_map.h"
#include "warglobals.h"
#include "gameanims.h"
#include "GfxManager.h"
#include "redrawqueue.h"
#include "gfxid.h"
#include "soundsamples.h"
#include "settings.h"
#include "animplayer.h"
#include "AnimId.h"
using namespace eng2d;

namespace WeWantWar {

///
/// Constants, datatypes and static methods
/// ============================================================================

const int Barrel::BURN_TIMER_INDEX;



///
/// Constructors, destructor and operators
/// ============================================================================

/** Default constructor.
 */
Barrel::Barrel( BarrelType aType ) :
  GameObject(),
  iBarrelType( aType ),
  iFlameAnimUid( -1 )
{
  ObjectID::Type oid = ObjectID::TYPE_BARREL;
  this->setCorrectAnimation( GameAnims::EIdle );
  this->boundingSphere( Settings::floatObjProp(oid, "bounding_sphere:") );
  this->setCollisionPoint(0, Vec2D(0 ,-28));
  this->setCollisionPoint(1, Vec2D(28, 0));
  this->setCollisionPoint(2, Vec2D(0 , 28));
  this->setCollisionPoint(3, Vec2D(-28,0));
  this->setArmor( Settings::floatObjProp(oid, "armor:") );
  this->setController( 0 );
  this->setProperties( GameObject::PROP_GRENADEFORCE );
  
  iShadowOffsetPos.set( Settings::floatObjProp(oid, "shadow_offset_x:"),
                        Settings::floatObjProp(oid, "shadow_offset_y:") );
}



/** Destructor.
 */
Barrel::~Barrel()
{
}



///
/// Methods inhertited from the base class(es)
/// ============================================================================

/** From GameObject, updates the object.
 */
void Barrel::update()
{
  if ( this->state() == GameObject::STATE_DYING ) {
    
    if ( this->getCounter( BURN_TIMER_INDEX ) < 0 ) {
      // This object has died.
      this->state( GameObject::STATE_KILLED );
      this->hidden( true );
      this->createExplosion();
      if ( iFlameAnimUid != -1 ) {
        AnimPlayer::removeAnimation( iFlameAnimUid );
      }
      return;
    }
    // Make sure the flame animation moves with us.
    AnimPlayer::AnimElement* anim = AnimPlayer::findAnim( iFlameAnimUid );
    if ( anim != 0 ) {
      anim->iPos = this->position();
    }
  }
}



/** From GameObject, Draws this object.
 */
void Barrel::redraw( RedrawQueue* aQueue )
{
  if ( this->hidden() == true ) {
    return;
  }
  Vec2D pos( this->position() );
  pos -= Vec2D( Map::scrollX, Map::scrollY );
  BITMAP* sprite = m_animation.currentFrame().asBitmap();
  pos -= Vec2D( sprite->w/2, sprite->h/2 );
  aQueue->add( RedrawQueue::PRI_NORMAL, pos.intX(), pos.intY(),
               RedrawQueue::SPTYPE_SPRITE, sprite );
               
  // Add shadow
  GraphicsContainer* barrelCombine = GfxManager::findGfxContainer( GfxId::KBarrel2 );
  if ( barrelCombine != 0 ) {
    Vec2D shadowPos = pos;
    shadowPos += iShadowOffsetPos;
    BITMAP* shadow = barrelCombine->GetBitmap(1);
    aQueue->add( RedrawQueue::PRI_BELOW_NORMAL,
                 shadowPos.intX(), shadowPos.intY(),
                 RedrawQueue::SPTYPE_SPRITE,
                 RedrawQueue::BMODE_MULTIPLY,
                 120, shadow );
  }
}



/** From GameObject, Kills this object
 */
void Barrel::kill()
{
  if ( this->state() != GameObject::STATE_LIVING ) {
    return;
  }
  
  this->state( GameObject::STATE_DYING );
  this->setCorrectAnimation( GameAnims::EDying );
  
  ObjectID::Type oid = ObjectID::TYPE_BARREL;
  int cval = Settings::intObjProp(oid, "burn_time:");
  cval += rand() % Settings::intObjProp(oid, "burn_time_variance:");
  this->setCounter( BURN_TIMER_INDEX, cval );
  
  // Spawn flame animation.
  const Animation& fireAnim = GameAnims::findAnimation( AnimId::KCampFire );
  iFlameAnimUid = AnimPlayer::spawn( fireAnim, this->position() );
}



/** From GameObject, Makes sound
 */
void Barrel::makeSound( GameObject::SoundID aSoundId ) const
{
  int sampleIndex = -1;
  switch ( aSoundId ) {
    case (GameObject::SND_PAIN): {
      sampleIndex = SMP_METALHIT;
      break;
    }
    case (GameObject::SND_DIE): {
      sampleIndex = SMP_GRENADE;
      break;
    }
    default: {
      sampleIndex = -1;
      break;
    }
  }
  if ( sampleIndex != -1 ) {
    Sound::playSample( sampleIndex, false );
  }
}



/** From GameObject, handles bullet hits
 */
bool Barrel::hitByBullet( Bullet* aBullet )
{
  this->causeDamage( aBullet );
  ParticleSparks* sparks = new ParticleSparks( aBullet->iPosition,
                                               aBullet->velocity(), 12 );
  WarGlobals::pPartManager->addSystem( sparks );
  this->makeSound( GameObject::SND_PAIN );
  return true;
}



/** From GameObject, Returns the type of this object.
 */
ObjectID::Type Barrel::objectType() const
{
  return ObjectID::TYPE_BARREL;
}



/** From GameObject, Tells if we're loading
 */
bool Barrel::reloading() const
{
  return false;
}



/** Sets the correct animation.
 */
void Barrel::setCorrectAnimation( int aAnimId )
{
  if ( iBarrelType == EOil ) {
    const Animation& anim = GameAnims::findAnimation( AnimId::KRedWasteBarrel, aAnimId );
    this->setAnimation( anim, aAnimId );
  }
}



/** Creates the explosion
 */
void Barrel::createExplosion()
{
  Sound::playSample(SMP_GRENADE, false);
  const Animation& lightAnim = GameAnims::findAnimation( AnimId::KGrenadeExplosionLight );
  if ( Settings::explosionLights == true ) {
    AnimPlayer::spawn( lightAnim, this->position(), 0 );
  }
  const Animation& anim = GameAnims::findAnimation( AnimId::KExplosionGrenade );
  AnimPlayer::spawn( anim, this->position(), 0 );
  float ac = 256.0 / 32.0;
  for (float a=0; a < 255; a+=ac) {
    eng2d::Vec2D spdVec( 0,-7 );
    spdVec.rotate( a );
    
    Bullet* b = BulletTable::createBullet( this, m_position,
                                           Bullet::EGrenade );
    b->setVelocity( spdVec );
    WarGlobals::pBulletManager->spawnBullet( b );
  }
}


};  // end of namespace
